<?php
/* --------------------------------------------------------------
  TitleCollection.php 2020-01-22
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2020 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace Gambio\ProductImageList\Image\Collections;

use ArrayAccess;
use Countable;
use Gambio\ProductImageList\Image\Exceptions\TextNotFoundException;
use Gambio\ProductImageList\Image\ValueObjects\AbstractText;
use Gambio\ProductImageList\Image\ValueObjects\LanguageCode;
use InvalidArgumentException;
use Iterator;
use JsonSerializable;

/**
 * Class TitleCollection
 * @package Gambio\ProductImageList\Image\Collections
 */
class TextCollection implements Countable, Iterator, ArrayAccess, JsonSerializable
{
    /**
     * @var AbstractText[]
     */
    protected $values = [];
    
    /**
     * @var int
     */
    protected $position = 0;
    
    
    /**
     * TitleCollection constructor.
     *
     * @param AbstractText[] $values
     */
    public function __construct(array $values = [])
    {
        if (count($values)) {
    
            foreach ($values as $text) {
                
                $this->addText($text);
            }
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function current()
    {
        return $this->values[$this->position];
    }
    
    
    /**
     * @inheritDoc
     */
    public function next()
    {
        $this->position++;
    }
    
    
    /**
     * @inheritDoc
     */
    public function key()
    {
        return $this->position;
    }
    
    
    /**
     * @inheritDoc
     */
    public function valid()
    {
        return isset($this->values[$this->position]);
    }
    
    
    /**
     * @inheritDoc
     */
    public function rewind()
    {
        $this->position = 0;
    }
    
    
    /**
     * @inheritDoc
     */
    public function offsetExists($offset)
    {
        return isset($this->values[$offset]);
    }
    
    
    /**
     * @inheritDoc
     */
    public function offsetGet($offset)
    {
        return $this->values[$offset];
    }
    
    
    /**
     * @inheritDoc
     */
    public function offsetSet($offset, $value)
    {
        if (!$value instanceof AbstractText) {
            
            throw new InvalidArgumentException(static::class . ' only accepts ' . AbstractText::class);
        }
    
        if (empty($offset)) {
            $this->values[] = $value;
        } else {
            $this->values[$offset] = $value;
        }
    }
    
    
    /**
     * @inheritDoc
     */
    public function offsetUnset($offset)
    {
        unset($this->values[$offset]);
    }
    
    
    /**
     * @inheritDoc
     */
    public function count()
    {
        return count($this->values);
    }
    
    /**
     * @param AbstractText $title
     */
    public function addText(AbstractText $title): void
    {
        $this->values[] = $title;
    }
    
    
    /**
     * @param LanguageCode $languageCode
     *
     * @return AbstractText
     * @throws TextNotFoundException
     */
    public function getTextByLanguageCode(LanguageCode $languageCode): AbstractText
    {
        foreach ($this as $text) {
        
            if ($text->languageCode()->value() === $languageCode->value()) {
                
                return $text;
            }
        }
        
        throw new TextNotFoundException('No text was found with the LanguageCode "' . $languageCode->value() . '"');
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize()
    {
        return $this->values;
    }
}